const express = require("express");
const multer = require("multer");
const path = require("path");
const {nanoid} = require("nanoid");
const config = require("../config");
const Product = require("../models/Product");
const auth = require("../middleware/auth");
const router = express.Router();

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, config.uploadPath);
  },
  filename: (req, file, cb) => {
    cb(null, nanoid() + path.extname(file.originalname));
  }
});

const upload = multer({storage});

const createRouter = () => {
  router.get("/", async (req, res) => {
    let category;
    if (req.query.category) {
      category = {category: req.query.category};
    }
    try {
      const products = await Product.find(category).populate("category", "title");
      res.send(products);
    } catch (e) {
      res.sendStatus(500);
    }
  });

  router.get("/:id", async (req, res) => {
    const product =  await Product.findById(req.params.id).populate('category author');
    res.send(product);
  });

  router.post("/", [upload.single("image"), auth], async (req, res) => {
    const productData = req.body;
    productData.author = req.user;

    if (req.file) {
      productData.image = req.file.filename;
    } else {
      productData.image = 'ic-dialog.png';
    }

    const product = new Product(productData);
    try {
      res.send(await product.save());
    } catch (err) {
      res.status(400).send(err);
    }
  });


  router.delete("/:id", auth, async (req, res) => {
    const product = await Product.findById(req.params.id);
    if (req.user._id.toString() === product.author.toString()) {
      await Product.findByIdAndRemove(req.params.id);
      return res.send(await Product.find());
    } else {
      return res.sendStatus(403);
    }
  });


  return router;
};


module.exports = createRouter;


