const mongoose = require("mongoose");
const bcrypt = require("bcrypt");
const {nanoid} = require("nanoid");
const Schema = mongoose.Schema;
const SALT_WORK_FACTOR = 10;

const UserSchema = new Schema({
  username: {
    type: String,
    required: true,
    unique: true,
    validate: {
      validator: async function(value) {
        if (!this.isModified("username")) return;
        const user = await User.findOne({username: value});
        if (user) return false;
      },
      message: props => `Username ${props.value} is already exists`
    }
  },
  password: {
    type: String,
    required: true
  },
  displayName: {
    type: String,
    required: true
  },
  phone: {
    type: String,
    required: true
  },
  token: String
});

UserSchema.pre("save", async function (next) {
  if (!this.isModified("password")) next();

  const salt = await bcrypt.genSalt(SALT_WORK_FACTOR);
  this.password = await bcrypt.hash(this.password, salt);
  next();
});

UserSchema.set("toJSON", {
  transform: (doc, ret) => {
    delete ret.password;
    return ret;
  }
});

UserSchema.methods.checkPassword = function(password) {
  return bcrypt.compare(password, this.password);
};
UserSchema.methods.generateToken = function() {
  this.token = nanoid();
};

const User = mongoose.model("User", UserSchema);

module.exports = User;
