﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Threading;
using RazorEngine;
using RazorEngine.Templating;
using Encoding = System.Text;
namespace MyHttpServer
{
    public class MyHttpServer
    {
         private Thread _serverThread;
        private string _siteDirectory;
        private HttpListener _listener;
        private int _port;

        public MyHttpServer(string path, int port)
        {
           this.Initialize(path, port); 
        }

        private void Initialize(string path, int port)
        {
            _siteDirectory = path;
            _port = port;
            _serverThread = new Thread(Listen);
            _serverThread.Start();
            Console.WriteLine($"Сервер запущен на порту: {port}");
            Console.WriteLine($"Файлы сайта лежат в папке {path}");
        }

        private void Listen()
        {
            _listener = new HttpListener();
            _listener.Prefixes.Add("http://localhost:" + _port.ToString() + "/");
            _listener.Start();
            while (true)
            {
                try
                {
                    HttpListenerContext context = _listener.GetContext();
                    Process(context);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.Message);
                    break;
                }
            }
            Stop();
        }

        private void Process(HttpListenerContext context)
        {
            string url = context.Request.Url.ToString();
            string filename = context.Request.Url.AbsolutePath;
            Console.WriteLine(filename);
            filename = filename.Substring(1);
            filename = Path.Combine(_siteDirectory, filename);

            if (File.Exists(filename))
            {
                try
                {
                    string content;
                    if (filename.Contains("html"))
                    {
                        content = BuildHtml(filename);
                    }
                    else
                    {
                        content = File.ReadAllText(filename);
                    }

                    byte[] htmlBytes = Encoding.Encoding.UTF8.GetBytes(content);
                    Stream fileStream = new MemoryStream(htmlBytes);
                    
                    context.Response.ContentType = GetContentType(filename);
                    context.Response.ContentLength64 = fileStream.Length;

                    byte[] buffer = new byte[16 * 1024];
                    int dataLength;
                    do
                    {
                        dataLength = fileStream.Read(buffer, 0, buffer.Length);
                        context.Response.OutputStream.Write(buffer, 0, dataLength);
                    } while (dataLength > 0);
                    fileStream.Close();
                    context.Response.StatusCode = (int) HttpStatusCode.OK;
                    context.Response.OutputStream.Flush();
                }
                catch (Exception e)
                {
                    Console.WriteLine(e);
                    context.Response.StatusCode = (int) HttpStatusCode.InternalServerError;
                }
            }
            else
            {
                context.Response.StatusCode = (int) HttpStatusCode.NotFound; 
            }
            context.Response.OutputStream.Close();
        }

        private string GetContentType(string filename)
        {
            var dictionary = new Dictionary<string, string>
            {
                {".css",  "text/css"},
                {".html", "text/html"},
                {".ico",  "image/x-icon"},
                {".js",   "application/x-javascript"},
                {".json", "application/json"},
                {".png",  "image/png"}
            };
            string contentType = "";
            string fileExension = Path.GetExtension(filename);
            dictionary.TryGetValue(fileExension, out contentType);
            return contentType;
        }
        
        
        

        public void Stop()
        {
            _serverThread.Abort();
            _listener.Stop();
        }

        private string BuildHtml(string fileName)
        {
            string html = "";
            string layoutPath = Path.Combine(_siteDirectory, "layout.html");
            string filePath = Path.Combine(_siteDirectory, fileName);

            var razorService = Engine.Razor;
            
            if(!razorService.IsTemplateCached("layout", null))
                razorService.AddTemplate("layout", File.ReadAllText(layoutPath));

            if (!razorService.IsTemplateCached(fileName, null))
            {
                razorService.AddTemplate(fileName, File.ReadAllText(filePath));
                razorService.Compile(fileName);
            }

            html = razorService.Run(fileName, null, new
            {
                IndexTitle = "Главная страница",
                Page1 = "Страница 1",
                Page2 = "Страница 2",
            });
            return html;
        }
        
        
    }
}