from django.shortcuts import render, redirect, get_object_or_404
from django.urls import reverse
from django.views import View
from django.views.generic import FormView

from .helpers.views import CustomFormView
from articles.models import Article, Author
from .forms import AuthorForm, CommentForm, ArticleForm


def article_list_view(request):
    if request.method == "GET":
        articles = Article.objects.all()
        return render(
            request,
            template_name="articles/article_list.html",
            context={'articles': articles}
        )


def author_list_view(request):
    if request.method == "GET":
        authors = Author.objects.all()
        return render(
            request,
            template_name="authors/author_list.html",
            context={'authors': authors}
        )


class CreateArticleView(CustomFormView):
    template_name = 'articles/create.html'
    form_class = ArticleForm

    def form_valid(self, form):
        data = {}
        tags = form.cleaned_data.pop('tags')
        for key, value in form.cleaned_data.items():
            if value is not None:
                data[key] = value

        self.article = Article.objects.create(**data)
        self.article.tags.set(tags)

        return super().form_valid(form)

    def get_redirect_url(self):
        return reverse('article_list')


class ArticleUpdateView(FormView):
    template_name = "articles/update.html"
    form_class = ArticleForm

    def dispatch(self, request, *args, **kwargs):
        self.article = self.get_object()
        return super().dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['article'] = self.article
        return context

    def get_initial(self):
        initial = {}
        for key in 'title', 'body', 'author', 'tags':
            initial[key] = getattr(self.article, key)
        initial['tags'] = self.article.tags.all()
        return initial

    def form_valid(self, form):
        tags = form.cleaned_data.pop('tags')
        for key, value in form.cleaned_data.items():
            if value is not None:
                setattr(self.article, key, value)
        self.article.save()
        self.article.tags.set(tags)
        return super().form_valid(form=form)

    def get_success_url(self):
        return reverse('article_detail', kwargs={'pk': self.article.pk})

    def get_object(self):
        pk = self.kwargs.get('pk')
        return get_object_or_404(Article, pk=pk)

class AuthorView(View):
    url_pattern = 'authors/author_create.html'

    def get(self, request, *args, **kwargs):
        form = AuthorForm()

        return render(
            request,
            self.url_pattern,
            context={'form': form})

    def post(self, request, *args, **kwargs):
        form = AuthorForm(request.POST)
        if form.is_valid():
            form.save()
        return redirect('author_list')


class ArticleDetailView(View):
    def get(self, request, *args, **kwargs):
        form = CommentForm()
        return render(request, "articles/detail.html", context={
            'article': get_object_or_404(Article, pk=kwargs.get("pk")),
            'form': form
        })


class CommentView(View):
    def post(self, request, *args, **kwargs):
        form = CommentForm(request.POST)
        article_pk = kwargs.get('article_pk')
        if form.is_valid():
            article = get_object_or_404(Article, pk=article_pk)
            article.comments.create(
                text=request.POST.get("text"),
                author=request.POST.get("author")
            )

        return redirect('article_detail', article_pk)


def create_author_view(request):
    if request.method == "GET":
        form = AuthorForm()
        return render(
            request,
            'authors/author_create.html',
            context={'form': form})
    if request.method == "POST":
        form = AuthorForm(request.POST)
        if form.is_valid():
            form.save()
        return redirect('author_list')


def author_edit_view(request, pk):
    author = get_object_or_404(Author, pk=pk)
    if request.method == "GET":
        form = AuthorForm(instance=author)
        return render(
            request,
            'authors/author_update.html',
            context={'form': form, "pk": pk})
    if request.method == "POST":
        form = AuthorForm(request.POST)
        if form.is_valid():
            author.name = request.POST.get("name")
            author.save(update_fields=['name', ])
        return redirect('author_list')


def author_delete_view(request, pk):
    if request.method == "GET":
        author = get_object_or_404(Author, pk=pk)
        author.delete()
        return redirect('author_list')
