from django.contrib.auth import (
    authenticate, login, logout, get_user_model
)
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib.auth.models import Group
from django.core.exceptions import ValidationError
from django.core.paginator import Paginator
from django.shortcuts import render, redirect
from django.views import View
from django.views.generic import DetailView

from .forms import RegisterForm, MyRegisterForm
from .models import Profile


class LogoutView(View):
    def get(self, request, *args, **kwargs):
        logout(request)
        next = request.GET.get('next')
        if next:
            return redirect(next)
        return redirect('login')


class LoginView(View):
    def get(self, request, *args, **kwargs):
        return render(
            request=request,
            template_name='login.html',
        )

    def post(self, request, *args, **kwargs):
        context = {}
        username = request.POST.get("username")
        password = request.POST.get("pass")
        user = authenticate(
            request,
            username=username,
            password=password
        )
        if user:
            login(request, user)
            next = request.GET.get('next')
            if next:
                return redirect(next)
            return redirect('article_list')
        else:
            context['has_error'] = True
        return render(
            request=request,
            template_name='login.html',
            context=context
        )


class RegisterView(View):
    def get(self, request, *args, **kwargs):
        form = MyRegisterForm()
        return render(
            request,
            template_name='register.html',
            context={"form": form})

    def post(self, request, *args, **kwargs):
        try:
            form = MyRegisterForm(data=request.POST)
            if form.is_valid():
                user = form.save()
                Profile.objects.create(user=user)
                group = Group.objects.get(name="Новые пользователи")
                group.user_set.add(user)
                login(request, user)
                return redirect('article_list')
        except ValidationError as e:
            print(e)
        return render(
            request,
            template_name='register.html',
            context={"form": form})


class UserDetailView(LoginRequiredMixin, DetailView):
    model = get_user_model()
    template_name = "user_detail.html"
    context_object_name = "user_obj"
    paginate_related_by = 5
    paginate_related_orphans = 0

    def get_context_data(self, **kwargs):
        articles = self.object.articles.order_by('-created_at')
        paginator = Paginator(articles, self.paginate_related_by, self.paginate_related_orphans)
        page_number = self.request.GET.get('page', 1)
        page = paginator.get_page(page_number)
        kwargs['page_obj'] = page
        kwargs['articles'] = page.object_list
        kwargs['is_paginated'] = page.has_other_pages()
        return super(UserDetailView, self).get_context_data(**kwargs)
