from django.db.models import Q
from django.shortcuts import render, redirect, get_object_or_404
from django.template.defaultfilters import urlencode
from django.urls import reverse
from django.views import View
from django.views.generic import FormView

from .helpers.views import CustomFormView, ListView
from articles.models import Article, Author
from .forms import AuthorForm, CommentForm, ArticleForm, SearchForm


class ArticleListView(ListView):
    form = SearchForm
    search_fields = ('title', 'name')
    template_name = 'articles/list.html'
    model = Article
    context_object_name = 'articles'
    ordering = ['-created_at']
    paginate_by = 5
    paginate_orphans = 1

    def get(self, request, *args, **kwargs):
        self.form = self.get_search_form()
        self.search_value = self.get_search_value()
        return super().get(request, *args, **kwargs)

    def get_context_data(self, *, object_list=None, **kwargs):
        context = super().get_context_data(object_list=object_list, **kwargs)
        context['form'] = self.form
        if self.search_value:
            context['query'] = urlencode({
                'search': self.search_value
            })
        return context

    def get_search_form(self):
        return self.form(self.request.GET)

    def get_search_value(self):
        if self.form.is_valid():
            return self.form.cleaned_data.get('search')

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.search_value:
            query = Q(title__icontains=self.search_value) | \
                    Q(author__name__icontains=self.search_value)
            queryset = queryset.filter(query)
        return queryset


class AuthorListView(ListView):
    template_name = 'authors/list.html'
    model = Author
    context_object_name = 'authors'
    ordering = ['name']
    paginate_by = 5


class CreateArticleView(CustomFormView):
    template_name = 'articles/create.html'
    form_class = ArticleForm

    def form_valid(self, form):
        data = {}
        tags = form.cleaned_data.pop('tags')
        for key, value in form.cleaned_data.items():
            if value is not None:
                data[key] = value

        self.article = Article.objects.create(**data)
        self.article.tags.set(tags)

        return super().form_valid(form)

    def get_redirect_url(self):
        return reverse('article_list')


class ArticleUpdateView(FormView):
    template_name = "articles/update.html"
    form_class = ArticleForm

    def dispatch(self, request, *args, **kwargs):
        self.article = self.get_object()
        return super().dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['article'] = self.article
        return context

    def get_initial(self):
        initial = {}
        for key in 'title', 'body', 'author', 'tags':
            initial[key] = getattr(self.article, key)
        initial['tags'] = self.article.tags.all()
        return initial

    def form_valid(self, form):
        tags = form.cleaned_data.pop('tags')
        for key, value in form.cleaned_data.items():
            if value is not None:
                setattr(self.article, key, value)
        self.article.save()
        self.article.tags.set(tags)
        return super().form_valid(form=form)

    def get_success_url(self):
        return reverse('article_detail', kwargs={'pk': self.article.pk})

    def get_object(self):
        pk = self.kwargs.get('pk')
        return get_object_or_404(Article, pk=pk)


class AuthorView(View):
    url_pattern = 'authors/create.html'

    def get(self, request, *args, **kwargs):
        form = AuthorForm()

        return render(
            request,
            self.url_pattern,
            context={'form': form})

    def post(self, request, *args, **kwargs):
        form = AuthorForm(request.POST)
        if form.is_valid():
            form.save()
        return redirect('author_list')


class ArticleDetailView(View):
    def get(self, request, *args, **kwargs):
        form = CommentForm()
        return render(request, "articles/detail.html", context={
            'article': get_object_or_404(Article, pk=kwargs.get("pk")),
            'form': form
        })


class CommentView(View):
    def post(self, request, *args, **kwargs):
        form = CommentForm(request.POST)
        article_pk = kwargs.get('article_pk')
        if form.is_valid():
            article = get_object_or_404(Article, pk=article_pk)
            article.comments.create(
                text=request.POST.get("text"),
                author=request.POST.get("author")
            )

        return redirect('article_detail', article_pk)


def create_author_view(request):
    if request.method == "GET":
        form = AuthorForm()
        return render(
            request,
            'authors/create.html',
            context={'form': form})
    if request.method == "POST":
        form = AuthorForm(request.POST)
        if form.is_valid():
            form.save()
        return redirect('author_list')


def author_edit_view(request, pk):
    author = get_object_or_404(Author, pk=pk)
    if request.method == "GET":
        form = AuthorForm(instance=author)
        return render(
            request,
            'authors/update.html',
            context={'form': form, "pk": pk})
    if request.method == "POST":
        form = AuthorForm(request.POST)
        if form.is_valid():
            author.name = request.POST.get("name")
            author.save(update_fields=['name', ])
        return redirect('author_list')


def author_delete_view(request, pk):
    if request.method == "GET":
        author = get_object_or_404(Author, pk=pk)
        author.delete()
        return redirect('author_list')
