from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.shortcuts import render, redirect
from django.views import View
from django.views.generic import TemplateView, ListView


class CustomFormView(View):
    form_class = None
    template_name = None
    redirect_url = ''

    def get(self, request, *args, **kwargs):
        form = self.form_class()

        context = self.get_context_data(form=form)
        return render(request, self.template_name, context=context)

    def post(self, request, *args, **kwargs):
        form = self.form_class(data=request.POST)
        if form.is_valid():
            return self.form_valid(form)
        return self.form_invalid(form)

    def form_valid(self, form):
        return redirect(self.get_redirect_url())

    def form_invalid(self, form):
        context = self.get_context_data(form=form)
        return render(self.request, self.template_name, context=context)

    def get_redirect_url(self):
        return self.redirect_url

    def get_context_data(self, **kwargs):
        return kwargs


class ListView(TemplateView):
    model = None
    context_object_name = 'objects'
    paginate_by = 5
    paginate_orphans = 0
    ordering = []

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        objects = self.get_queryset().order_by(*self.ordering)
        paginator = Paginator(object_list=objects, per_page=self.paginate_by, orphans=self.paginate_orphans)

        page = self.request.GET.get('page')

        try:
            objects_list = paginator.page(page)
        except PageNotAnInteger:
            objects_list = paginator.page(1)
        except EmptyPage:
            objects_list = paginator.page(paginator.num_pages)

        context['page_obj'] = objects_list
        context[self.context_object_name] = objects_list.object_list

        return context

    def get_queryset(self):
        return self.model.objects.all()
