package com.example.final_exam_l.controller;

import com.example.final_exam_l.dto.add.PlaceAddDTO;
import com.example.final_exam_l.dto.add.ReviewAddDTO;
import com.example.final_exam_l.entity.Place;
import com.example.final_exam_l.entity.PlacePhoto;
import com.example.final_exam_l.repository.PlaceRepository;
import com.example.final_exam_l.service.*;
import lombok.Getter;
import org.springframework.core.io.Resource;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.security.Principal;
import java.util.stream.Collectors;

@Controller
@RequiredArgsConstructor
@RequestMapping
public class PlaceController {
    private final PlaceService service;
    private final UserService userService;
    private final FileSystemStorageService fileSystemStorageService;
    private final PlacePhotoService placePhotoService;
    private final PropertiesService propertiesService;
    private final StorageService storageService;
    private final ReviewService reviewService;

    @GetMapping("/{id}")
    public String one(@PathVariable Integer id, Model model){
        model.addAttribute("place", service.findOne(id));
        return "place/place";
    }

    @GetMapping
    public String all(Model model, Pageable pageable, HttpServletRequest uriBuilder){
        Page<Place> places = service.all(pageable);
        model.addAttribute("places", places);
        String uri = uriBuilder.getRequestURI();
        PageableView.constructPageable(places, propertiesService.getDefaultPageSize(), model, uri);
        return "place/places";
    }

    @RequestMapping(path = "/", params = "search", method = RequestMethod.GET)
    public String all(@RequestParam String search, Pageable pageable, HttpServletRequest uriBuilder,
                      Model model){
        Page<Place> places = service.filter(pageable, search);
        model.addAttribute("places", places);
        String uri = uriBuilder.getRequestURI();
        PageableView.constructPageable(places, propertiesService.getDefaultPageSize(), model, uri);
        return "place/places";
    }

    @PreAuthorize("hasAuthority('USER') || hasAuthority('ADMIN')")
    @GetMapping("/add")
    public String add(){
        return "place/add";
    }

    @PreAuthorize("hasAuthority('USER') || hasAuthority('ADMIN')")
    @PostMapping("/add")
    public String add(@Valid PlaceAddDTO dto,
                      BindingResult validationResult,
                      RedirectAttributes attributes,
                      @RequestParam("file") MultipartFile file,
                      Principal principal){
        attributes.addFlashAttribute("dto", dto);
        if (validationResult.hasFieldErrors()) {
            attributes.addFlashAttribute("errors", validationResult.getFieldErrors());
            return "redirect:/add";
        }
        else if (service.isUnique(dto.getName())) {
            attributes.addFlashAttribute("nameError", "Заведение с названием "  + dto.getName() + " уже существует");
            return "redirect:/add";
        }
        if (!service.isPhoto(file) || file.isEmpty()){
            attributes.addFlashAttribute("fileError", "Загрузите фото в формате png или jpg");
            return "redirect:/add";
        }
        Place place = service.add(dto, principal.getName());
        PlacePhoto placePhoto = placePhotoService.add(principal.getName(), file.getOriginalFilename(), place);
        place.setPlacePhoto(placePhoto);
        service.save(place);
        fileSystemStorageService.store(file);
        return "redirect:/" + place.getId();
    }

    @PreAuthorize("hasAuthority('ADMIN')")
    @PostMapping("/{id}/delete")
    public String delete(@PathVariable Integer id){
        service.delete(id);
        return "redirect:/";
    }

    @PreAuthorize("hasAuthority('USER') || hasAuthority('ADMIN')")
    @PostMapping("/{id}/review")
    public String addReview(@PathVariable Integer id,
                            @Valid ReviewAddDTO dto,
                            BindingResult validationResult,
                            RedirectAttributes attributes,
                            Principal principal){
        attributes.addFlashAttribute("dto", dto);
        if (validationResult.hasFieldErrors()) {
            attributes.addFlashAttribute("errors", validationResult.getFieldErrors());
            return "redirect:/" + id;
        }
        else if (reviewService.isUnique(id, userService.findByEmail(principal.getName()).getId())) {
            attributes.addFlashAttribute("nameError", "Вы уже оставляли отзыв к данному заведению.");
            return "redirect:/" + id;
        }
        reviewService.add(dto, principal.getName(), service.findOne(id));
        return "redirect:/" + id;
    }

    @PreAuthorize("hasAuthority('ADMIN')")
    @PostMapping("/{id}/review/{reviewId}")
    public String deleteReview(@PathVariable Integer id, @PathVariable Integer reviewId){
        reviewService.delete(reviewId, id);
        return "redirect:/" + id;
    }

    @PreAuthorize("hasAuthority('ADMIN')")
    @PostMapping("/{id}/photo/{photoId}")
    public String deletePhoto(@PathVariable Integer id, @PathVariable Integer photoId){
        placePhotoService.delete(photoId);
        return "redirect:/" + id;
    }

    @PreAuthorize("hasAuthority('USER') || hasAuthority('ADMIN')")
    @PostMapping("/{id}/photo")
    public String addPhoto(@PathVariable Integer id,
                            RedirectAttributes attributes,
                            Principal principal,
                            @RequestParam("file") MultipartFile file){
        if (!service.isPhoto(file) || file.isEmpty()){
            attributes.addFlashAttribute("fileError", "Загрузите фото в формате png или jpg");
            return "redirect:/add";
        }
        placePhotoService.add(principal.getName(), file.getOriginalFilename(), service.findOne(id));
        return "redirect:/" + id;
    }

    @GetMapping("/file/{filename:.+}")
    public ResponseEntity<Resource> getFilePic(@PathVariable String filename) {
        {
            Resource file = storageService.loadAsResource(filename);
            return ResponseEntity.ok().header(HttpHeaders.CONTENT_DISPOSITION,
                            "attachment; filename=\"" + file.getFilename() + "\"")
                    .body(file);
        }
    }
}
