import { Button, Flex, Form, Input } from 'antd'
import { useForm, type FormProps } from 'antd/es/form/Form'
import { useEffect, useRef, useState } from 'react'

interface Message {
	username: string
	text: string
}

export type FieldType = {
	username: string
	message: string
}

const boxStyle: React.CSSProperties = {
	width: '100%',
	height: 120,
}

export function ChatApp() {
	const [messages, setMessages] = useState<Message[]>([])
	const [username, setUsername] = useState('')
	const [isLoggedIn, setIsLoggedIn] = useState(false)

	const [form] = useForm()

	const ws = useRef<WebSocket | null>(null)

	useEffect(() => {
		ws.current = new WebSocket('ws://localhost:8000/chat')

		ws.current.onclose = () => console.log('ws closed')

		ws.current.onmessage = (event: MessageEvent<string>) => {
			const decodedMessage: { type: string; message: Message } = JSON.parse(
				event.data
			)

			if (decodedMessage.type === 'NEW_MESSAGE') {
				setMessages(prev => {
					return [...prev, decodedMessage.message]
				})
			}
		}

		return () => ws.current?.close()
	}, [])

	const submitHandlerUsername: FormProps<FieldType>['onFinish'] = values => {
		ws.current?.send(
			JSON.stringify({
				type: 'SET_USERNAME',
				username: values.username,
			})
		)

		setUsername(values.username)
		setIsLoggedIn(true)
	}

	const submitMessageHandler = (values: { message: string }) => {
		ws.current?.send(
			JSON.stringify({
				type: 'CREATE_MESSAGE',
				text: values.message,
				username,
			})
		)

		console.log(
			JSON.stringify({
				type: 'CREATE_MESSAGE',
				text: values.message,
				username,
			})
		)

		form.resetFields()
	}

	return !isLoggedIn ? (
		<Flex justify='center' align='center' style={boxStyle}>
			<Form name='login' onFinish={submitHandlerUsername} autoComplete='off'>
				<Form.Item<FieldType>
					name='username'
					rules={[{ required: true, message: 'Please input your username!' }]}
				>
					<Input placeholder='Enter username' />
				</Form.Item>

				<Form.Item wrapperCol={{ offset: 8, span: 16 }}>
					<Button type='primary' htmlType='submit'>
						Submit
					</Button>
				</Form.Item>
			</Form>
		</Flex>
	) : (
		<>
			<div>Welcome, {username}</div>
			{messages.map((message, key) => (
				<div key={key}>
					<b>{message.username}</b>: {message.text}
				</div>
			))}

			<Flex justify='center' align='center' style={boxStyle}>
				<Form
					form={form}
					name='messageForm'
					onFinish={submitMessageHandler}
					autoComplete='off'
				>
					<Form.Item<FieldType>
						name='message'
						rules={[{ required: true, message: 'Please enter message' }]}
					>
						<Input placeholder='Enter message' />
					</Form.Item>

					<Form.Item wrapperCol={{ offset: 8, span: 16 }}>
						<Button type='primary' htmlType='submit'>
							Submit
						</Button>
					</Form.Item>
				</Form>
			</Flex>
		</>
	)
}
