import { useEffect, useCallback, useState, useMemo  } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import MonthCalendarBody from '../../components/MonthCalendar/MonthCalendarBody/MonthCalendarBody';
import MonthCalendarHeader from '../../components/MonthCalendar/MonthCalendarHeader/MonthCalendarHeader';
import { dateToISOLikeButLocal, getCurrentMonthString, getDaysInMonth } from '../../helpers/CalendarHelpers';
import { addCalendarTask, addCopyCalendarTask, deleteCalendarTask, editCalendarTask, fetchCalendarTasks} from '../../store/actions/tasksActions';

function MonthCalendar() {
  const dispatch = useDispatch();
  const { calendarTasks } = useSelector(state => state.tasks);
  const user = useSelector(state => state.users?.user);

  const [hourFormat, setHourFormat] = useState(false);
  const [dateNow, setDateNow] = useState({month: '', year: ''})
  const [worker, setWorker] = useState('');
  const [calendarType, setCalendarType] = useState('Месяц');
  const [currentTask, setCurrentTask] = useState({title: '', description: '', priority: null, infoForCell: {startHour: null, endHour: null}})
  const [copyTask, setCopyTask] = useState(null)
  const [cellSizes, setCellSizes] = useState({})

  useEffect(()=>{
    setDateNow({
      month: new Date().getMonth(),
      year: new Date().getFullYear(),
    })
    dispatch(fetchCalendarTasks())
  }, [dispatch])


  const hoursInDay = useMemo(()=>{
    let arr
    if (hourFormat) {
      arr = ['8:00', '9:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00', '18:00', '19:00', '20:00','21:00','22:00']
    } else {
      arr = ['8:00', '10:00', '12:00', '14:00', '16:00', '18:00', '20:00', '22:00']
    }
    const cells = arr.length
    const xs = 10.8/cells
    setCellSizes(()=>{
      return {smallCell: 0.6, standarCell: xs, dayCell: 12/cells}
    })
    return arr
  }, [hourFormat])

  const daysInMonth = useMemo(() => {
    return getDaysInMonth(dateNow)
  }, [dateNow])

  const currentMonthString = useMemo(() => {
    return getCurrentMonthString(dateNow.month)
  }, [dateNow.month])

  const onChangeWorkerHandler = useCallback((event) => {
    setWorker(event.target.value);
  }, []);

  const onChangeCalendarTypeHandler = useCallback((event) => {
    setCalendarType(event.target.value);
  }, []);

  const incrementMonth = useCallback(() => {
    setDateNow((prevState)=>{
      if (prevState.month + 1 === 12 ) { 
        return { month: 0, year: prevState.year + 1}
      }
      return {...prevState, month: prevState.month + 1}
    })
  }, [])
  const decrementMonth = useCallback(() => {
    setDateNow((prevState)=>{
      if (prevState.month - 1 === -1) {
        return  { month: 11, year: prevState.year - 1}
      }
      return {...prevState, month: prevState.month - 1}
    })
  }, [])

  const onChangeCurrentTaskHandler = useCallback((e) => {
    const {name, value} = e.target;
    if (name === 'startHour' || name === 'endHour') {
      setCurrentTask((prevState) => {
        return {
            ...prevState,
            infoForCell: {
              ...prevState.infoForCell,
              [name]: parseInt(value) 
            }
        }
      });
    } else {
      setCurrentTask((prevState) => {
        return {
            ...prevState,
            [name]: value
        }
      });
    }
  }, []);

  const createTaskInCellHandler = (dayNumber, dayHour) => {
    let hour
    if (!isNaN(dayHour)) {
      hour = dayHour
    } else {
      hour = parseInt(dayHour.split(':')[0])
    }
    let hourDue
    if (hourFormat) {
      hourDue = hour + 1
    } else {
      hourDue = hour + 2
    }
    const newTask = {
      title:"Задача",
      description:"описание",
      priority: null,
      dateTimeStart: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0)),
      dateTimeDue: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hourDue, 59)),
      infoForCell: {
        startHour: hour,
        endHour: hourDue,
        startDay: dayNumber
      }
    }
    setCurrentTask((newTask))
  }
  
  const dragTaskHandler = async (dayNumber, hour) => {
    const hourDiff = currentTask.infoForCell.endHour - currentTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...currentTask,
      dateTimeTaskId: currentTask.id,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newTask.id
    delete newTask.infoForCell
    await dispatch(editCalendarTask(newTask))
    setCurrentTask({})
  }

  const sendNewTaskHandler = async () => {
    const timeEndHour = currentTask.infoForCell.endHour
    const timeStartHour = currentTask.infoForCell.startHour
    const day = currentTask.infoForCell.startDay
    const due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, day, timeEndHour - 1, 59))
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, day, timeStartHour, 0))
    if (currentTask.id) {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: currentTask.mainTaskId
      }
      delete newTask.infoForCell
      await dispatch(editCalendarTask(newTask, currentTask.id))
    } else {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        executor: user,
        dateTimeDeadLine: due,
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCalendarTask(newTask))
    }
  }

  const createCopyTask = async (dayNumber, hour) => {
    const hourDiff = copyTask.infoForCell.endHour - copyTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...copyTask,
      dateTimeStart: start,
      dateTimeDue: due,
      taskId: copyTask.mainTaskId
    }
    delete newTask.infoForCell
    delete newTask.id
    await dispatch(addCopyCalendarTask(newTask))
    setCopyTask(null)
  }

  const deleteTaskHandler = async (taskId) => {
    dispatch(deleteCalendarTask(taskId))
  }

  return (
    <>
      <MonthCalendarHeader
        year={dateNow.year}
        currentMonthString={currentMonthString} 
        decrementMonth={decrementMonth}
        incrementMonth={incrementMonth}
        onChangeCalendarTypeHandler={onChangeCalendarTypeHandler}
        onChangeWorkerHandler={onChangeWorkerHandler}
        worker={worker}
        calendarType={calendarType}
      />
      <MonthCalendarBody
        month={dateNow.month}
        year={dateNow.year}
        tasks={calendarTasks}
        createTaskInCellHandler={createTaskInCellHandler}
        setCurrentTask={setCurrentTask}
        hourFormat={hourFormat} 
        setHourFormat={setHourFormat}
        currentTask={currentTask}
        onChangeCurrentTaskHandler={onChangeCurrentTaskHandler}
        sendNewTaskHandler={sendNewTaskHandler}
        deleteTaskHandler={deleteTaskHandler}
        cellSizes={cellSizes}
        hoursInDay={hoursInDay}
        daysInMonth={daysInMonth}
        dragTaskHandler={dragTaskHandler}
        createCopyTask={createCopyTask}
        copyTask={copyTask}
        setCopyTask={setCopyTask}
      />

    </>
  );
}

export default MonthCalendar;
