import { useEffect, useCallback, useState, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import CalendarModalWorkerContent from '../../components/Calendars/UI/CalendarModalWorkerContent/CalendarModalWorkerContent';
import MonthCalendarBody from '../../components/Calendars/MonthCalendar/MonthCalendarBody/MonthCalendarBody';
import MonthCalendarHeader from '../../components/Calendars/MonthCalendar/MonthCalendarHeader/MonthCalendarHeader';
import DefaultModal from '../../components/UI/DefaultModal/DefaultModal';
import { dateToISOLikeButLocal, getCurrentMonthString, getDaysInMonth } from '../../helpers/CalendarHelpers';
import { fetchAllUserProjects, fetchProject } from '../../store/actions/projectsActions';
import { addCalendarTask, addCopyCalendarTask, deleteCalendarTask, editCalendarTask, fetchCalendarTasks } from '../../store/actions/tasksActions';
import { fetchCurrentCalendarDisplayName } from '../../store/actions/usersActions';
import { AllHoursOneHourFormat, AllHoursTwoHourFormat } from '../../constants';

function MonthCalendar() {
  const dispatch = useDispatch();

  const { calendarTasks } = useSelector(state => state.tasks);
  const { user, currentCalendarDisplayName } = useSelector(state => state.users);
  const { allUserProjects, project } = useSelector(state => state.projects)

  const [hourFormat, setHourFormat] = useState(false);
  const [dateNow, setDateNow] = useState({ month: '', year: '' })
  const [workerInfo, setWorkerInfo] = useState({ project: '', worker: '' });
  const [currentTask, setCurrentTask] = useState({ title: '', description: '', priority: null, infoForCell: { startHour: null, endHour: null } })
  const [copyTask, setCopyTask] = useState(null)
  const [cellSizes, setCellSizes] = useState({})
  const [userId, setUserId] = useState('')
  const [userCalendarId, setUserCalendarId] = useState(null)
  const [modal, setModal] = useState(false)

  useEffect(() => {
    setDateNow({
      month: new Date().getMonth(),
      year: new Date().getFullYear(),
    })
  }, [dispatch])

  useEffect(() => {
    if (userCalendarId) {
      setUserId(userCalendarId)
      dispatch(fetchCalendarTasks(userCalendarId))
      dispatch(fetchCurrentCalendarDisplayName(userCalendarId))
      dispatch(fetchAllUserProjects())
    } else {
      setUserId(user.id)
      dispatch(fetchCalendarTasks(user.id))
      dispatch(fetchCurrentCalendarDisplayName(user.id))
      dispatch(fetchAllUserProjects())
    }
  }, [userCalendarId, dispatch, user.id])

  useEffect(() => {
    if (workerInfo.project) {
      dispatch(fetchProject(workerInfo.project.id))
    }
  }, [workerInfo.project, dispatch])

  const hoursInDay = useMemo(() => {
    let arr
    if (hourFormat) {
      arr = AllHoursOneHourFormat
    } else {
      arr = AllHoursTwoHourFormat
    }
    const cells = arr.length
    const xs = 10.8 / cells
    setCellSizes(() => {
      return { smallCell: 0.6, standarCell: xs, dayCell: 12 / cells }
    })
    return arr
  }, [hourFormat])

  const daysInMonth = useMemo(() => {
    return getDaysInMonth(dateNow)
  }, [dateNow])

  const currentMonthString = useMemo(() => {
    return getCurrentMonthString(dateNow.month)
  }, [dateNow.month])

  const onChangeProjectHandler = useCallback((e, value) => {
    setWorkerInfo((prevState) => { return { ...prevState, project: value } });
  }, []);

  const onChangeWorkerHandler = useCallback((e, value) => {
    setWorkerInfo((prevState) => { return { ...prevState, worker: value } });
  }, []);

  const incrementMonth = useCallback(() => {
    setDateNow((prevState) => {
      if (prevState.month + 1 === 12) {
        return { month: 0, year: prevState.year + 1 }
      }
      return { ...prevState, month: prevState.month + 1 }
    })
  }, [])

  const decrementMonth = useCallback(() => {
    setDateNow((prevState) => {
      if (prevState.month - 1 === -1) {
        return { month: 11, year: prevState.year - 1 }
      }
      return { ...prevState, month: prevState.month - 1 }
    })
  }, [])

  const onChangeCurrentTaskHandler = useCallback((e) => {
    const { name, value } = e.target;
    if (name === 'startHour' || name === 'endHour') {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          infoForCell: {
            ...prevState.infoForCell,
            [name]: value
          }
        }
      });
    } else {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          [name]: value
        }
      });
    }
  }, []);

  const createTaskInCellHandler = useCallback((dayNumber, dayHour) => {
    let hour
    if (!isNaN(dayHour)) {
      hour = dayHour
    } else {
      hour = parseInt(dayHour.split(':')[0])
    }
    let hourDue
    if (hourFormat) {
      hourDue = hour + 1
    } else {
      hourDue = hour + 2
    }
    const newTask = {
      title: "Задача",
      description: "описание",
      priority: null,
      dateTimeStart: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0)),
      dateTimeDue: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hourDue, 59)),
      infoForCell: {
        startHour: hour,
        endHour: hourDue,
        startDay: dayNumber
      },
      project: allUserProjects[0].id
    }
    setCurrentTask((newTask))
  }, [dateNow.month, dateNow.year, hourFormat, allUserProjects])

  const dragTaskHandler = useCallback(async (dayNumber, hour) => {
    const hourDiff = currentTask.infoForCell.endHour - currentTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...currentTask,
      taskId: currentTask.mainTaskId,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newTask.id
    delete newTask.infoForCell
    await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    setCurrentTask({})
  }, [currentTask, dateNow.month, dateNow.year, dispatch, hoursInDay, userId])

  const sendNewTaskHandler = useCallback(async () => {
    const timeEndHour = currentTask.infoForCell.endHour
    const timeStartHour = currentTask.infoForCell.startHour
    const day = currentTask.infoForCell.startDay
    const due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, day, timeEndHour - 1, 59))
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, day, timeStartHour, 0))
    if (currentTask.id) {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: currentTask.mainTaskId
      }
      delete newTask.infoForCell
      await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    } else {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        executor: userId,
        author: user.id,
        dateTimeDeadLine: due,
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCalendarTask(newTask, userId))
    }
  }, [currentTask, dateNow.month, dateNow.year, dispatch, user.id, userId])

  const createCopyTask = useCallback(async (dayNumber, hour) => {
    const hourDiff = copyTask.infoForCell.endHour - copyTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...copyTask,
      dateTimeStart: start,
      dateTimeDue: due,
      taskId: copyTask.mainTaskId
    }
    delete newTask.infoForCell
    delete newTask.id
    await dispatch(addCopyCalendarTask(newTask, userId))
    setCopyTask(null)
  }, [copyTask, dateNow.month, dateNow.year, dispatch, hoursInDay, userId])

  const deleteTaskHandler = useCallback(async (taskId) => {
    dispatch(deleteCalendarTask(taskId, userId))
  }, [dispatch, userId])

  const handleClose = useCallback(() => {
    setModal(false)
    setWorkerInfo({ project: '', worker: '' })
  }, [])

  const handleOpen = useCallback(async () => {
    if (user.id === userId) {
      await dispatch(fetchAllUserProjects())
      setModal(true)
    } else {
      setUserCalendarId(null)
    }
  }, [dispatch, user.id, userId])

  const onChangeCalendarUser = useCallback(() => {
    setModal(false)
    setUserCalendarId(workerInfo.worker.user.id)
    setWorkerInfo({ project: '', worker: '' })
  }, [workerInfo?.worker?.user?.id])

  return (
    <>
      <DefaultModal
        modal={modal}
        handleClose={() => { handleClose() }}
      >
        <CalendarModalWorkerContent
          workerInfo={workerInfo}
          allUserProjects={allUserProjects}
          workers={project?.project?.members}
          onChangeProjectHandler={onChangeProjectHandler}
          onChangeWorkerHandler={onChangeWorkerHandler}
          handleClose={handleClose}
          onChangeCalendarUser={onChangeCalendarUser}
        />
      </DefaultModal>

      <MonthCalendarHeader
        year={dateNow.year}
        currentMonthString={currentMonthString}
        decrementMonth={decrementMonth}
        incrementMonth={incrementMonth}
        handleOpen={handleOpen}
        currentCalendarDisplayName={currentCalendarDisplayName}
        user={user}
        userId={userId}
      />
      <MonthCalendarBody
        month={dateNow.month}
        year={dateNow.year}
        tasks={calendarTasks}
        createTaskInCellHandler={createTaskInCellHandler}
        setCurrentTask={setCurrentTask}
        hourFormat={hourFormat}
        setHourFormat={setHourFormat}
        currentTask={currentTask}
        onChangeCurrentTaskHandler={onChangeCurrentTaskHandler}
        sendNewTaskHandler={sendNewTaskHandler}
        deleteTaskHandler={deleteTaskHandler}
        cellSizes={cellSizes}
        hoursInDay={hoursInDay}
        daysInMonth={daysInMonth}
        dragTaskHandler={dragTaskHandler}
        createCopyTask={createCopyTask}
        copyTask={copyTask}
        setCopyTask={setCopyTask}
        allUserProjects={allUserProjects}
      />

    </>
  );
}

export default MonthCalendar;
