import { useEffect, useCallback, useState, useMemo  } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import MonthCalendarBody from '../../components/MonthCalendarBody/MonthCalendarBody';
import MonthCalendarHeader from '../../components/MonthCalendarHeader/MonthCalendarHeader';
import { dateToISOLikeButLocal, getCurrentMonthString, getDaysInMonth } from '../../helpers/CalendarHelpers';
import { addTask, deleteTask, editTask, fetchCalendarTasks} from '../../store/actions/tasksActions';

function MonthCalendar() {
  const dispatch = useDispatch();
  const { calendarTasks } = useSelector(state => state.tasks);

  const [hourFormat, setHourFormat] = useState(false);
  const [dateNow, setDateNow] = useState({month: '', year: ''})
  const [worker, setWorker] = useState('');
  const [calendarType, setCalendarType] = useState('Месяц');
  const [currentTask, setCurrentTask] = useState({title: '', description: '', priority: ''})
  const [copyTask, setCopyTask] = useState(null)
  const [cellSizes, setCellSizes] = useState({})

  useEffect(()=>{
    setDateNow({
      month: new Date().getMonth(),
      year: new Date().getFullYear(),
    })
    dispatch(fetchCalendarTasks())
  }, [dispatch])


  const hoursInDay = useMemo(()=>{
    let arr
    if (hourFormat) {
      arr = ['8:00', '9:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00', '18:00', '19:00', '20:00','21:00','22:00']
    } else {
      arr = ['8:00', '10:00', '12:00', '14:00', '16:00', '18:00', '20:00', '22:00']
    }
    const cells = arr.length
    const xs = 10.8/cells
    setCellSizes(()=>{
      return {smallCell: 0.6, standarCell: xs, dayCell: 12/cells}
    })
    return arr
  }, [hourFormat])

  const daysInMonth = useMemo(() => {
    return getDaysInMonth(dateNow)
  }, [dateNow])

  const currentMonthString = useMemo(() => {
    return getCurrentMonthString(dateNow)
  }, [dateNow])

  const onChangeWorkerHandler = useCallback((event) => {
    setWorker(event.target.value);
  }, []);

  const onChangeCalendarTypeHandler = useCallback((event) => {
    setCalendarType(event.target.value);
  }, []);

  const incrementMonth = useCallback(() => {
    setDateNow((prevState)=>{
      if (prevState.month + 1 === 12 ) { 
        return { month: 0, year: prevState.year + 1}
      }
      return {...prevState, month: prevState.month + 1}
    })
  }, [])

  const decrementMonth = useCallback(() => {
    setDateNow((prevState)=>{
      if (prevState.month - 1 === -1) {
        return  { month: 11, year: prevState.year - 1}
      }
      return {...prevState, month: prevState.month - 1}
    })
  }, [])

  const onChangeCurrentTaskHandler = useCallback((e) => {
    const {name, value} = e.target;
    setCurrentTask((prevState) => {
        return {
            ...prevState,
            [name]: value
        }
    });
  }, []);

  const createTaskInCellHandler = (dayNumber, dayHour) => {
    let hour
    if (!isNaN(dayHour)) {
      hour = dayHour
    } else {
      hour = parseInt(dayHour.split(':')[0])
    }
    let hourDue
    if (hourFormat) {
      hourDue = hour + 0
    } else {
      hourDue = hour + 1
    }
    const newTask = {
      title:"Задача",
      description:"описание",
      priority: null,
      dateTimeStart: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0)),
      dateTimeDue: dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hourDue, 59)),
    }
    setCurrentTask((newTask))
  }
  
  const dragTaskHandler = async (dayNumber, hour) => {
    const hourDiff = currentTask.infoForCell.endHour - currentTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newObj = {
      ...currentTask,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newObj.infoForCell
    await dispatch(editTask(newObj))
    setCurrentTask({})
  }

  const increaseTaskHandler = async (dayNumber, task, isStartTask) => {
    const timeEndHour = task.infoForCell.endHour
    const timeStartHour = task.infoForCell.startHour
    let hourDiff
    if (hourFormat) {
      hourDiff = 1
    } else {
      hourDiff = 2
    }
    let due
    let start 
    if (isStartTask) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeEndHour, 59))
      start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeStartHour - hourDiff, 0))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeEndHour + hourDiff, 59))
      start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeStartHour, 0))
    }
    const newObj = {
      ...task,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newObj.infoForCell
    await dispatch(editTask(newObj))
    setCurrentTask({})

  }

  const reduceTaskHandler = async (dayNumber, task, isStartTask) => {
    const timeEndHour = task.infoForCell.endHour
    const timeStartHour = task.infoForCell.startHour
    let hourDiff
    let hourDiffCheck
    if (hourFormat) {
      hourDiff = 1
      hourDiffCheck = 0
    } else {
      hourDiff = 2
      hourDiffCheck = 1
    }
    let due
    let start
    if (task.infoForCell.endHour - task.infoForCell.startHour - hourDiffCheck !== 0) {
      if (isStartTask) {
        due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeEndHour, 59))
        start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeStartHour + hourDiff, 0))
      } else {
        due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeEndHour - hourDiff, 59))
        start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, timeStartHour, 0))
      }
      const newObj = {
        ...task,
        dateTimeStart: start,
        dateTimeDue: due
      }
      delete newObj.infoForCell
      await dispatch(editTask(newObj))
      setCurrentTask({})
    }
  }

  const sendNewTaskHandler = async () => {
    if (currentTask.id) {
      delete currentTask.infoForCell
      setCurrentTask(() => {
        return{
        ...currentTask,
      }}
      )
      await dispatch(editTask(currentTask))
    } else {
      setCurrentTask(() => {
        return{
        ...currentTask,
      }}
      )
      delete currentTask.infoForCell
      await dispatch(addTask(currentTask))
    }
  }

  const createCopyTask = async (dayNumber, hour) => {
    const hourDiff = copyTask.infoForCell.endHour - copyTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...copyTask,
      dateTimeStart: start,
      dateTimeDue: due,
    }
    delete newTask.infoForCell
    delete newTask.id
    await dispatch(addTask(newTask))
    setCopyTask(null)
  }

  const deleteTaskHandler = async (taskId) => {
    dispatch(deleteTask(taskId))
  }

  return (
    <>
      <MonthCalendarHeader
        year={dateNow.year}
        currentMonthString={currentMonthString} 
        decrementMonth={decrementMonth}
        incrementMonth={incrementMonth}
        onChangeCalendarTypeHandler={onChangeCalendarTypeHandler}
        onChangeWorkerHandler={onChangeWorkerHandler}
        worker={worker}
        calendarType={calendarType}
      />
      <MonthCalendarBody
        month={dateNow.month}
        year={dateNow.year}
        tasks={calendarTasks}
        createTaskInCellHandler={createTaskInCellHandler}
        setCurrentTask={setCurrentTask}
        hourFormat={hourFormat} 
        setHourFormat={setHourFormat}
        currentTask={currentTask}
        onChangeCurrentTaskHandler={onChangeCurrentTaskHandler}
        sendNewTaskHandler={sendNewTaskHandler}
        deleteTaskHandler={deleteTaskHandler}
        cellSizes={cellSizes}
        hoursInDay={hoursInDay}
        daysInMonth={daysInMonth}
        dragTaskHandler={dragTaskHandler}
        increaseTaskHandler={increaseTaskHandler}
        reduceTaskHandler={reduceTaskHandler}
        createCopyTask={createCopyTask}
        copyTask={copyTask}
        setCopyTask={setCopyTask}
      />

    </>
  );
}

export default MonthCalendar;
