import moment from 'moment';
import { useEffect, useState, useMemo, useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import WeekCalendarBody from '../../components/Calendars/WeekCalendar/WeekCalendarBody/WeekCalendarBody';
import WeekCalendarHeader from '../../components/Calendars/WeekCalendar/WeekCalendarHeader/WeekCalendarHeader';
import { getWeekInfoString, getWeekFromCurrentDate, dateToISOLikeButLocal } from '../../helpers/CalendarHelpers';
import { addCalendarTask, addCopyCalendarTask, deleteCalendarTask, editCalendarTask, fetchCalendarTasks } from '../../store/actions/tasksActions';


function WeekCalendar() {
  const dispatch = useDispatch();
  const { calendarTasks } = useSelector(state => state.tasks);
  const [copyTask, setCopyTask] = useState(null)
  const user = useSelector(state => state.users?.user);
  const [dateNow, setDateNow] = useState({ year: '', month: '', currentDay: '' })
  const [currentTask, setCurrentTask] = useState({ title: '', description: '', priority: null, infoForCell: { startHour: null, endHour: null } })
  const [hourFormat, setHourFormat] = useState(false);
  const [userId, setUserId] = useState('')

  useEffect(() => {
    const year = new Date().getFullYear()
    const month = new Date().getMonth()
    const currentDay = moment().date()
    setDateNow({ year: year, month: month, currentDay: currentDay })
    dispatch(fetchCalendarTasks(user.id))
    setUserId(user.id)
  }, [dispatch, user.id])
  
  const hoursInDay = useMemo(() => {
    let arr
    if (hourFormat) {
      arr = ['8:00', '9:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00', '18:00', '19:00', '20:00', '21:00', '22:00']
    } else {
      arr = ['8:00', '10:00', '12:00', '14:00', '16:00', '18:00', '20:00', '22:00']
    }
    return arr
  }, [hourFormat])

  const week = useMemo(() => {
    return getWeekFromCurrentDate(dateNow.year, dateNow.month, dateNow.currentDay)
  }, [dateNow])

  const weekInfo = useMemo(() => {
    return getWeekInfoString(week, dateNow)
  }, [dateNow, week])

  const incrementWeek = useCallback(() => {
    setDateNow((prevState) => {
      const newDate = new Date(prevState.year, prevState.month, prevState.currentDay + 7)
      return { year: newDate.getFullYear(), month: newDate.getMonth(), currentDay: moment(newDate).date() }
    })
  }, [])
  const decrementWeek = useCallback(() => {
    setDateNow((prevState) => {
      const newDate = new Date(prevState.year, prevState.month, prevState.currentDay - 7)
      return { year: newDate.getFullYear(), month: newDate.getMonth(), currentDay: moment(newDate).date() }
    })
  }, [])
  const onChangeCurrentTaskHandler = useCallback((e) => {
    const { name, value } = e.target;
    if (name === 'startHour' || name === 'endHour') {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          infoForCell: {
            ...prevState.infoForCell,
            [name]: value
          }
        }
      });
    } else {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          [name]: value
        }
      });
    }
  }, []);

  const createTaskInCellHandler = useCallback((dayNumber, dayHour, month) => {
    let hour
    if (!isNaN(dayHour)) {
      hour = dayHour
    } else {
      hour = parseInt(dayHour.split(':')[0])
    }
    let hourDue
    if (hourFormat) {
      hourDue = hour + 1
    } else {
      hourDue = hour + 2
    }
    const newTask = {
      title: "Задача",
      description: "описание",
      priority: null,
      dateTimeStart: dateToISOLikeButLocal(new Date(dateNow.year, month, dayNumber, hour, 0)),
      dateTimeDue: dateToISOLikeButLocal(new Date(dateNow.year, month, dayNumber, hourDue, 59)),
      infoForCell: {
        startHour: hour,
        endHour: hourDue,
        startDay: dayNumber,
        month: month
      }
    }
    setCurrentTask((newTask))
  }, [dateNow.month, dateNow.year, hourFormat])

  const sendNewTaskHandler = useCallback(async () => {
    const timeEndHour = currentTask.infoForCell.endHour
    const timeStartHour = currentTask.infoForCell.startHour
    const month = currentTask.infoForCell.month
    const day = currentTask.infoForCell.startDay
    const due = dateToISOLikeButLocal(new Date(dateNow.year, month, day, timeEndHour - 1, 59))
    const start = dateToISOLikeButLocal(new Date(dateNow.year, month, day, timeStartHour, 0))
    if (currentTask.id) {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: currentTask.mainTaskId
      }
      delete newTask.infoForCell
      await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    } else {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        executor: userId,
        author: user.id,
        dateTimeDeadLine: due,
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCalendarTask(newTask, userId))
    }
  }, [currentTask, dateNow.month, dateNow.year, dispatch, user.id, userId])

  const createCopyTask = useCallback(async (dayNumber, hour, month) => {
    const hourDiff = copyTask.infoForCell.endHour - copyTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, month, dayNumber, hour, 0))
    const newTask = {
      ...copyTask,
      dateTimeStart: start,
      dateTimeDue: due,
      taskId: copyTask.mainTaskId
    }
    delete newTask.infoForCell
    delete newTask.id
    await dispatch(addCopyCalendarTask(newTask, userId))
    setCopyTask(null)
  }, [copyTask, dateNow.month, dateNow.year, dispatch, hoursInDay, userId])

  const dragTaskHandler = useCallback(async (dayNumber, hour) => {
    const hourDiff = currentTask.infoForCell.endHour - currentTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(dateNow.year, dateNow.month, dayNumber, hour, 0))
    const newTask = {
      ...currentTask,
      taskId: currentTask.mainTaskId,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newTask.id
    delete newTask.infoForCell
    await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    setCurrentTask({})
  }, [currentTask, dateNow.month, dateNow.year, dispatch, hoursInDay, userId])

  const deleteTaskHandler = useCallback(async (taskId) => {
    dispatch(deleteCalendarTask(taskId, userId))
  },[dispatch, userId])

  return (
    <>
      <WeekCalendarHeader
        incrementWeek={incrementWeek}
        decrementWeek={decrementWeek}
        weekInfo={weekInfo}
      />
      <WeekCalendarBody
        deleteTaskHandler={deleteTaskHandler}
        sendNewTaskHandler={sendNewTaskHandler}
        onChangeCurrentTaskHandler={onChangeCurrentTaskHandler}
        currentTask={currentTask}
        setCurrentTask={setCurrentTask}
        tasks={calendarTasks}
        date={dateNow}
        week={week}
        hourFormat={hourFormat}
        setHourFormat={setHourFormat}
        hoursInDay={hoursInDay}
        createTaskInCellHandler={createTaskInCellHandler}
        createCopyTask={createCopyTask}
        copyTask={copyTask}
        setCopyTask={setCopyTask}
        dragTaskHandler={dragTaskHandler}
      />
    </>
  );
}

export default WeekCalendar;