import {
  Column,
  Entity,
  PrimaryGeneratedColumn,
  CreateDateColumn,
  BeforeInsert,
  BaseEntity,
  OneToMany,
  OneToOne,
} from 'typeorm';
import {IsEmail
} from "class-validator";
import { Exclude, instanceToPlain } from "class-transformer";
import bcrypt from 'bcrypt';
import {nanoid} from 'nanoid';
import {Task} from './Task';
import {Member} from './Member';
import { PasswordRecovery } from './PasswordRecovery';

export const SALT_WORK_FACTOR= 10;


export enum UserRole  {USER="user" ,DIRECTOR= "director",SUPERUSER="superuser"}

interface IUser {
  id:string;
  role: UserRole;
  name: string;
  surname: string;
  email: string;
  displayName: string;
  password:string;
  token: string;
  createdAt: Date;
  createdTasks:Task[];
  members: Member[];
}


@Entity({ name: 'User' })
export class User extends BaseEntity implements IUser {
  @PrimaryGeneratedColumn("uuid")
  id!: string;
  @Column({ name: 'name', type: 'varchar', length:30,nullable: false })
  name!: string;

  @Column({ name: 'surname', type: 'varchar', length:30,nullable: false })
  surname!: string;

  @Column({ name: 'displayName', type: 'varchar', length:35,nullable: false })
  displayName!: string;

  @Column({ name: 'email', type: 'varchar',length:40, unique: true, nullable: false })
  @IsEmail()
  email!: string;

  @Column({ name: 'phone', type: 'varchar',length:15, unique: true, nullable: true})
  phone?: string;

  @Column({ name: 'token', type: 'varchar',length:100, unique: true, nullable: false })
  token!: string;
  
  @CreateDateColumn({ name: 'created_at', type: Date, default: new Date() })
  createdAt!: Date;
  
  @Column({
    type: "enum",
    enum: UserRole,
    default: UserRole.USER
  })
  role!: UserRole
  
  @Column({ type: 'varchar', nullable: false, select:true })
  @Exclude({ toPlainOnly: true })
  password!: string;

  @OneToMany(() => Task, (task: { user: User }) => task.user)
  createdTasks!: Task[];

  @OneToMany(() => Task, (task: { user: User }) =>task.user)
  tasks!: Task[];

  @OneToMany(() => Member, (member: { user: User }) => member.user)
  members!: Member[];



  @BeforeInsert()
  protected async beforeInserthashPassword():Promise<void>  {
    const salt = await bcrypt.genSalt(SALT_WORK_FACTOR);
    this.password = await bcrypt.hash(this.password, salt);
    
  }

  public generateToken():void{
    this.token = nanoid()
    return
  }

  public async checkPassword(
    candidatePassword: string,
  ):Promise<boolean> {
    return await bcrypt.compare(candidatePassword, this.password);
  }
  
  toJSON() {
    return instanceToPlain(this);
}



}
















