import moment from 'moment';
import { useEffect, useState, useMemo, useCallback } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';

import CreateCopyTasksModeBlock from '../../components/Calendars/CreateCopyTasksModeBlock/CreateCopyTasksModeBlock';
import CalendarModalWorkerContent from '../../components/Calendars/UI/CalendarModalWorkerContent/CalendarModalWorkerContent';
import WeekCalendarBody from '../../components/Calendars/WeekCalendar/WeekCalendarBody/WeekCalendarBody';
import WeekCalendarHeader from '../../components/Calendars/WeekCalendar/WeekCalendarHeader/WeekCalendarHeader';
import DefaultModal from '../../components/UI/DefaultModal/DefaultModal';
import { AllHoursOneHourFormat, AllHoursTwoHourFormat } from '../../constants';
import { getWeekInfoString, getWeekFromCurrentDate, dateToISOLikeButLocal } from '../../helpers/CalendarHelpers';
import { fetchAllUserProjects } from '../../store/actions/projectsActions';
import { addCalendarTask, addCopyCalendarTask, deactivateCreateCopyTasksMode, deleteCalendarTask, editCalendarTask, fetchCalendarTasks } from '../../store/actions/tasksActions';
import { fetchCurrentCalendarDisplayName } from '../../store/actions/usersActions';

function WeekCalendar() {
  const dispatch = useDispatch();
  const navigate = useNavigate()
  const { calendarTasks, copyMode } = useSelector(state => state.tasks);
  const { user, currentCalendarDisplayName } = useSelector(state => state.users);
  const { allUserProjects } = useSelector(state => state.projects)
  console.log(copyMode)
  const [weekGoal, setWeekGoal] = useState('Наладить режим сна')
  const [weekPriorities, setWeekPriorities] = useState({ priorityOne: 'Один', priorityTwo: 'Два', priorityThree: 'Три' })
  const [workerInfo, setWorkerInfo] = useState({ project: '', worker: '' });
  const [dateNow, setDateNow] = useState({ year: '', month: '', currentDay: '' })
  const [currentTask, setCurrentTask] = useState({ title: '', description: '', priority: null, infoForCell: { startHour: null, endHour: null } })
  const [hourFormat, setHourFormat] = useState(false);
  const [copyTask, setCopyTask] = useState(null)
  const [userCalendarId, setUserCalendarId] = useState(null)
  const [modal, setModal] = useState(false)
  const [userId, setUserId] = useState('')
  console.log(copyTask)
  useEffect(() => {
    const year = new Date().getFullYear()
    const month = new Date().getMonth()
    const currentDay = moment().date()
    setDateNow({ year: year, month: month, currentDay: currentDay })
    return () => {
      dispatch(deactivateCreateCopyTasksMode())
    }
  }, [dispatch, user.id])

  useEffect(() => {
    if (userCalendarId) {
      setUserId(userCalendarId)
      dispatch(fetchCalendarTasks(userCalendarId))
      dispatch(fetchCurrentCalendarDisplayName(userCalendarId))
      dispatch(fetchAllUserProjects())
    } else {
      setUserId(user.id)
      dispatch(fetchCalendarTasks(user.id))
      dispatch(fetchCurrentCalendarDisplayName(user.id))
      dispatch(fetchAllUserProjects())
    }
  }, [userCalendarId, dispatch, user.id])

  const hoursInDay = useMemo(() => {
    let arr
    if (hourFormat) {
      arr = AllHoursOneHourFormat
    } else {
      arr = AllHoursTwoHourFormat
    }
    return arr
  }, [hourFormat])

  const week = useMemo(() => {
    return getWeekFromCurrentDate(dateNow.year, dateNow.month, dateNow.currentDay)
  }, [dateNow])

  const weekInfo = useMemo(() => {
    return getWeekInfoString(week, dateNow)
  }, [dateNow, week])

  const incrementWeek = useCallback(() => {
    setDateNow((prevState) => {
      const newDate = new Date(prevState.year, prevState.month, prevState.currentDay + 7)
      return { year: newDate.getFullYear(), month: newDate.getMonth(), currentDay: moment(newDate).date() }
    })
  }, [])

  const decrementWeek = useCallback(() => {
    setDateNow((prevState) => {
      const newDate = new Date(prevState.year, prevState.month, prevState.currentDay - 7)
      return { year: newDate.getFullYear(), month: newDate.getMonth(), currentDay: moment(newDate).date() }
    })
  }, [])

  const onChangeProjectHandler = useCallback((e, value) => {
    setWorkerInfo((prevState) => { return { ...prevState, project: value } });
  }, []);

  const onChangeWorkerHandler = useCallback((e, value) => {
    setWorkerInfo((prevState) => { return { ...prevState, worker: value } });
  }, []);

  const onChangeWeekGoalHandler = useCallback((e) => {
    setWeekGoal((prevState) => {
      return e.target.value
    })
  }, [])

  const onChangeWeekPrioritiesHandler = useCallback((e) => {
    const { name, value } = e.target;
    setWeekPriorities((prevState) => {
      return {
        ...prevState,
        [name]: value
      }
    })
  }, [])


  const onChangeCurrentTaskHandler = useCallback((e) => {
    const { name, value } = e.target;
    if (name === 'startHour' || name === 'endHour') {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          infoForCell: {
            ...prevState.infoForCell,
            [name]: value
          }
        }
      });
    } else {
      setCurrentTask((prevState) => {
        return {
          ...prevState,
          [name]: value
        }
      });
    }
  }, []);

  const onChangeCalendarUser = useCallback(() => {
    setModal(false)
    setUserCalendarId(workerInfo.worker.user.id)
    setWorkerInfo({ project: '', worker: '' })
  }, [workerInfo?.worker?.user?.id])

  const createTaskInCellHandler = useCallback((dayNumber, dayHour, month, year) => {
    let hour
    if (!isNaN(dayHour)) {
      hour = dayHour
    } else {
      hour = parseInt(dayHour.split(':')[0])
    }
    let hourDue
    if (hourFormat) {
      hourDue = hour + 1
    } else {
      hourDue = hour + 2
    }
    const newTask = {
      title: "Задача",
      description: "описание",
      priority: null,
      dateTimeStart: dateToISOLikeButLocal(new Date(year, month, dayNumber, hour, 0)),
      dateTimeDue: dateToISOLikeButLocal(new Date(year, month, dayNumber, hourDue, 59)),
      infoForCell: {
        startHour: hour,
        endHour: hourDue,
        startDay: dayNumber,
        month: month,
        startYear: year,
      },
      project: allUserProjects[0]?.id
    }
    setCurrentTask((newTask))
  }, [hourFormat, allUserProjects])

  const sendNewTaskHandler = useCallback(async () => {
    const timeEndHour = currentTask.infoForCell.endHour
    const timeStartHour = currentTask.infoForCell.startHour
    const month = currentTask.infoForCell.month
    const day = currentTask.infoForCell.startDay
    const year = currentTask.infoForCell.startYear
    const due = dateToISOLikeButLocal(new Date(year, month, day, timeEndHour - 1, 59))
    const start = dateToISOLikeButLocal(new Date(year, month, day, timeStartHour, 0))
    if (currentTask.id) {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: currentTask.mainTaskId
      }
      delete newTask.infoForCell
      await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    } else {
      const newTask = {
        ...currentTask,
        dateTimeStart: start,
        dateTimeDue: due,
        executor: userId,
        author: user.id,
        dateTimeDeadLine: due,
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCalendarTask(newTask, userId))
    }
  }, [currentTask, dispatch, user.id, userId])

  const createCopyTask = useCallback(async (dayNumber, hour, month, year) => {
    if (copyMode.working) {
      const hourDiff = hourFormat ? 0 : 1
      const due = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour + hourDiff, 59))
      const start = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour, 0))
      const newTask = {
        ...copyMode.task,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: copyMode.task.id
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCopyCalendarTask(newTask, userId))
    } else {
      const hourDiff = copyTask.infoForCell.endHour - copyTask.infoForCell.startHour
      const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
      let due
      if (hour + hourDiff >= lastHour) {
        due = dateToISOLikeButLocal(new Date(year, month, dayNumber, lastHour, 59))
      } else {
        due = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour + hourDiff, 59))
      }
      const start = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour, 0))
      const newTask = {
        ...copyTask,
        dateTimeStart: start,
        dateTimeDue: due,
        taskId: copyTask.mainTaskId
      }
      delete newTask.infoForCell
      delete newTask.id
      await dispatch(addCopyCalendarTask(newTask, userId))
      setCopyTask(null)
    }
  }, [copyTask, dispatch, hoursInDay, userId, copyMode.task, copyMode.working, hourFormat])

  const dragTaskHandler = useCallback(async (dayNumber, hour, month, year) => {
    const hourDiff = currentTask.infoForCell.endHour - currentTask.infoForCell.startHour
    const lastHour = hoursInDay[hoursInDay.length - 1].split(':')[0]
    let due
    if (hour + hourDiff >= lastHour) {
      due = dateToISOLikeButLocal(new Date(year, month, dayNumber, lastHour, 59))
    } else {
      due = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour + hourDiff, 59))
    }
    const start = dateToISOLikeButLocal(new Date(year, month, dayNumber, hour, 0))
    const newTask = {
      ...currentTask,
      taskId: currentTask.mainTaskId,
      dateTimeStart: start,
      dateTimeDue: due
    }
    delete newTask.id
    delete newTask.infoForCell
    await dispatch(editCalendarTask(newTask, currentTask.id, userId))
    setCurrentTask({})
  }, [currentTask, dispatch, hoursInDay, userId])

  const deleteTaskHandler = useCallback(async (e, taskId) => {
    e.stopPropagation();
    await dispatch(deleteCalendarTask(taskId, userId))
  }, [dispatch, userId])

  const handleClose = useCallback(() => {
    setModal(false)
    setWorkerInfo({ project: '', worker: '' })
  }, [])

  const handleOpen = useCallback(async () => {
    if (user.id === userId) {
      await dispatch(fetchAllUserProjects())
      setModal(true)
    } else {
      setUserCalendarId(null)
    }
  }, [dispatch, user.id, userId])

  const copyModeStay = useCallback(async () => {
    dispatch(deactivateCreateCopyTasksMode())
  }, [dispatch])

  const copyModeReturn = useCallback(async () => {
    navigate('/my-tasks')
  }, [navigate])

  return (
    <>
      <DefaultModal
        modal={modal}
        handleClose={() => { handleClose() }}
      >
        <CalendarModalWorkerContent
          workerInfo={workerInfo}
          allUserProjects={allUserProjects}
          workers={workerInfo?.project?.members}
          onChangeProjectHandler={onChangeProjectHandler}
          onChangeWorkerHandler={onChangeWorkerHandler}
          handleClose={handleClose}
          onChangeCalendarUser={onChangeCalendarUser}
        />
      </DefaultModal>

      <WeekCalendarHeader
        incrementWeek={incrementWeek}
        decrementWeek={decrementWeek}
        weekInfo={weekInfo}
        weekGoal={weekGoal}
        onChangeWeekGoalHandler={onChangeWeekGoalHandler}
        weekPriorities={weekPriorities}
        onChangeWeekPrioritiesHandler={onChangeWeekPrioritiesHandler}
        handleOpen={handleOpen}
        currentCalendarDisplayName={currentCalendarDisplayName}
        user={user}
        userId={userId}
      />
      <WeekCalendarBody
        deleteTaskHandler={deleteTaskHandler}
        sendNewTaskHandler={sendNewTaskHandler}
        onChangeCurrentTaskHandler={onChangeCurrentTaskHandler}
        currentTask={currentTask}
        setCurrentTask={setCurrentTask}
        tasks={calendarTasks}
        date={dateNow}
        week={week}
        hourFormat={hourFormat}
        setHourFormat={setHourFormat}
        hoursInDay={hoursInDay}
        createTaskInCellHandler={createTaskInCellHandler}
        createCopyTask={createCopyTask}
        copyTask={copyTask}
        setCopyTask={setCopyTask}
        dragTaskHandler={dragTaskHandler}
        copyMode={copyMode}
      />

      <CreateCopyTasksModeBlock
        copyMode={copyMode.working}
        copyModeStay={() => { copyModeStay() }}
        copyModeReturn={() => { copyModeReturn() }}
      />
    </>
  );
}

export default WeekCalendar;