import { Grid } from "@mui/material";
import { useNavigate, useParams } from "react-router-dom";
import { useSelector, useDispatch } from "react-redux";
import { useCallback, useEffect, useMemo, useState } from "react";
import { changeMemberRole, createMember, deleteMember, fetchProject } from "../../store/actions/projectsActions";
import { fetchUsers } from "../../store/actions/usersActions";
import ProjectUsersColumnsWrapper from "../../components/ProjectComponents/ProjectUsersColumnsWrapper/ProjectUsersColumnsWrapper";
import ProjectInfo from "../../components/ProjectComponents/ProjectInfo/ProjectInfo";
import DefaultModal from "../../components/UI/DefaultModal/DefaultModal"
import NewMemberModalContent from "../../components/ProjectComponents/NewMemberModalContent/NewMemberModalContent";



const FullProject = ({ projectId }) => {

    const { project } = useSelector(state => state.projects);
    const { user, users } = useSelector(state => state.users);

    const dispatch = useDispatch();
    const params = useParams()
    const navigate = useNavigate()

    const [modal, setModal] = useState(false)
    const [newMember, setNewMember] = useState(null)
    const [currentMember, setCurrentMember] = useState(null)


    useEffect(() => {
        if (projectId) {
            dispatch(fetchProject(projectId))
        } else {
            dispatch(fetchProject(params.id))
        }
        dispatch(fetchUsers())
    }, [params.id, dispatch, projectId]);

    const members = useMemo(() => {
        return project?.project?.members || []
    }, [project])

    const currentRoleInProject = useMemo(() => {
        return members.find((member) => member.user.id === user.id)?.roleProject
    }, [members, user.id])

    const onChangeRoleHandler = useCallback((e, value) => {
        setNewMember((prevState) => { return { ...prevState, roleProject: value?.value } });
    }, []);

    const onChangeNewMemberHandler = useCallback((e, value) => {
        setNewMember((prevState) => { return { ...prevState, userId: value?.id } });
    }, []);

    const handleOpen = useCallback(async () => {
        setModal(true)
    }, [])

    const handleClose = useCallback(() => {
        setModal(false)
        setNewMember(null)
    }, [])

    const createNewMemberHandler = useCallback(() => {
        dispatch(createMember({ ...newMember, projectId: project?.project?.id }))
        setModal(false)
        setNewMember(null)
    }, [dispatch, newMember, project?.project?.id])

    const dragMemberToNewRole = useCallback((role) => {
        if (currentMember?.id) {
            dispatch(changeMemberRole(currentMember?.id, { newRoleProject: role?.value, projectId: project?.project?.id }))
        }
    }, [currentMember, dispatch, project?.project?.id])

    const deleteMemberHandler = useCallback((id) => {
        dispatch(deleteMember(id, project?.project?.id))
    }, [dispatch, project?.project?.id])

    const onClickTasksHandler = useCallback(() => {
        navigate('/workers-tasks')
    }, [navigate])

    return <>
        <ProjectInfo
            project={project.project}
            handleOpen={handleOpen}
            currentRoleInProject={currentRoleInProject}
            onClickTasksHandler={onClickTasksHandler}
        />

        <ProjectUsersColumnsWrapper
            members={members}
            deleteMemberHandler={deleteMemberHandler}
            currentRoleInProject={currentRoleInProject}
            dragMemberToNewRole={dragMemberToNewRole}
            setCurrentMember={setCurrentMember}
        />

        <DefaultModal
            modal={modal}
            handleClose={() => { handleClose() }}
        >
            <NewMemberModalContent
                users={users}
                onChangeNewMemberHandler={onChangeNewMemberHandler}
                onChangeRoleHandler={onChangeRoleHandler}
                newMember={newMember}
                handleClose={handleClose}
                createNewMemberHandler={createNewMemberHandler}
                members={members}
            >

            </NewMemberModalContent>
        </DefaultModal>
    </>
};

export default FullProject;
