import {createAsyncThunk, createSlice} from '@reduxjs/toolkit';
import type {PayloadAction} from '@reduxjs/toolkit';
import axios from 'axios';
import IUser from '../../interfaces/IUser';
import {useNavigate} from 'react-router';

interface UserState {
	user: IUser;
	loading: boolean;
	userLoggedIn: boolean;
}

const initialState: UserState = {
	user: {} as IUser,
	loading: false,
	userLoggedIn: false,
};

export const createUser = createAsyncThunk(
	'createUser',
	async (user: IUser) => {
		try {
			await axios.post(`${import.meta.env.VITE_BASE_URL}/users/register`, {
				username: user.username,
				password: user.password,
			});
		} catch (err: any) {
			console.log(err);
		}
	}
);

export const loginUser = createAsyncThunk('loginUser', async (user: IUser) => {
	try {
		const response = await axios.post(
			`${import.meta.env.VITE_BASE_URL}/users/login`,
			{
				username: user.username,
				password: user.password,
			}
		);
		return response.data;
	} catch (err: any) {
		throw new Error(err);
	}
});

export const getUser = createAsyncThunk('getUser', async (token: string) => {
	try {
		const response = await axios.post(
			`${import.meta.env.VITE_BASE_URL}/users/getToken`,
			{token}
		);
		return response.data;
	} catch (err: any) {
		throw new Error(err);
	}
});

export const userSlice = createSlice({
	name: 'user',
	initialState,
	reducers: {
		setInitialUser: (state) => {
			state.user = {_id: '', username: '', password: '', token: ''};
			state.userLoggedIn = false;
		},
	},
	extraReducers: (builder) => {
		builder
			.addCase(createUser.pending, (state, action) => {
				state.loading = true;
			})
			.addCase(createUser.rejected, (state, action) => {
				state.loading = false;
			})
			.addCase(createUser.fulfilled, (state, action) => {
				state.loading = false;
			})
			.addCase(loginUser.pending, (state, action) => {
				state.loading = true;
			})
			.addCase(loginUser.rejected, (state, action) => {
				state.loading = false;
				console.log(action.error);
				state.userLoggedIn = false;
			})
			.addCase(
				loginUser.fulfilled,
				(state, {payload}: PayloadAction<{token: string}>) => {
					state.loading = false;
					state.userLoggedIn = true;
					window.sessionStorage.setItem('token', payload.token);
				}
			)
			.addCase(getUser.pending, (state, action) => {
				state.loading = true;
			})
			.addCase(getUser.rejected, (state, action) => {
				state.loading = false;
				state.userLoggedIn = false;
			})
			.addCase(getUser.fulfilled, (state, action) => {
				state.user.username = action.payload[0].username;
				state.loading = false;
				state.userLoggedIn = true;
				state.user._id = action.payload[0]._id;
			});
	},
});

export const {setInitialUser} = userSlice.actions;

export default userSlice.reducer;
