import {createAsyncThunk, createSlice} from '@reduxjs/toolkit';
import IPost from '../../interfaces/IPost';
import {postApi} from '../../api/postApi';
import IResponse from '../../interfaces/IResponse';

interface IPostState {
	posts: IPost[];
	post: IPost;
	loadingPosts: boolean;
	messagePosts: string;
}

const initialState: IPostState = {
	posts: [] as IPost[],
	post: {} as IPost,
	loadingPosts: false,
	messagePosts: '',
};
const namespace = 'posts';

export const getPosts = createAsyncThunk(
	`${namespace}/getPosts`,
	async (): Promise<IResponse<IPost[] | undefined>> => {
		return await postApi.getPosts();
	}
);

export const getPostById = createAsyncThunk(
	`${namespace}/getPostById`,
	async (id: string): Promise<IResponse<IPost | undefined>> => {
		return await postApi.getPostById(id);
	}
);

export const createPost = createAsyncThunk(
	`${namespace}/createPost`,
	async (post: FormData) => {
		return await postApi.createPost(post);
	}
);

export const deletePostById = createAsyncThunk(
	`${namespace}/deletePostById`,
	async (id: string) => {
		return await postApi.deletePostById(id);
	}
);

export const postsSlice = createSlice({
	name: namespace,
	initialState,
	reducers: {},
	extraReducers: (builder) => {
		builder
			.addCase(getPosts.rejected, (state) => {
				state.loadingPosts = false;
			})
			.addCase(getPosts.pending, (state) => {
				state.loadingPosts = true;
			})
			.addCase(getPosts.fulfilled, (state, action) => {
				state.loadingPosts = false;
				state.posts = action.payload.result as IPost[];
				state.messagePosts = action.payload.message;
			})
			.addCase(getPostById.rejected, (state) => {
				state.loadingPosts = false;
			})
			.addCase(getPostById.pending, (state) => {
				state.loadingPosts = true;
			})
			.addCase(getPostById.fulfilled, (state, action) => {
				state.loadingPosts = false;
				state.post = action.payload.result as IPost;
				state.messagePosts = action.payload.message;
			})
			.addCase(createPost.rejected, (state) => {
				state.loadingPosts = false;
			})
			.addCase(createPost.pending, (state) => {
				state.loadingPosts = true;
			})
			.addCase(createPost.fulfilled, (state, action) => {
				state.loadingPosts = false;
				state.messagePosts = action.payload.message;
				console.log(action.payload);
			})
			.addCase(deletePostById.rejected, (state) => {
				state.loadingPosts = false;
			})
			.addCase(deletePostById.pending, (state) => {
				state.loadingPosts = true;
			})
			.addCase(deletePostById.fulfilled, (state, action) => {
				state.loadingPosts = false;
				state.messagePosts = action.payload.message;
			});
	},
});

export default postsSlice.reducer;
