import { createAsyncThunk, createSlice } from '@reduxjs/toolkit'
import IResponse from '../../interfaces/IResponse'
import IComment from '../../interfaces/IComment'
import { commentsApi } from '../../api/commentApi'
import ICommentDto from '../../interfaces/ICommentDto'

interface CommentState {
    comments: IComment[];
    loading: boolean;
}

const initialState: CommentState = {
	comments: [] as IComment[],
    loading: false
};

export const getCommentsByPost = createAsyncThunk(
    'getCommentsByPost',
    async (postId: string): Promise<IResponse<IComment[] | undefined>> => {
        return commentsApi.getCommentsByPost(postId)
    }
)

export const createComment = createAsyncThunk(
    'createComment',
    async (comment: ICommentDto) => {
        return commentsApi.createComment(comment)
    }
)

export const deleteCommentById = createAsyncThunk(
    'deleteCommentById',
    async (id: string) => {
        return commentsApi.deleteCommentById(id)
    }
)

export const commentsSlice = createSlice({
    name: 'comment',
    initialState,
    reducers: {
        setInitialComment: (state) => {
            state.comments = [];
            state.loading = false
        }
    },
    extraReducers: (builder) => {
        builder
            .addCase(getCommentsByPost.rejected, (state) => {
                state.loading = false
            })
            .addCase(getCommentsByPost.pending, (state) => {
                state.loading = true
            })
            .addCase(getCommentsByPost.fulfilled, (state, action) => {
                state.loading = false
                state.comments = action.payload.result as IComment[]
            })
            .addCase(createComment.rejected, (state) => {
                state.loading = false
            })
            .addCase(createComment.pending, (state) => {
                state.loading = true
            })
            .addCase(createComment.fulfilled, (state, action) => {
                state.loading = false
            })
            .addCase(deleteCommentById.rejected, (state) => {
                state.loading = false
            })
            .addCase(deleteCommentById.pending, (state) => {
                state.loading = true
            })
            .addCase(deleteCommentById.fulfilled, (state, action) => {
                state.loading = false
            })
    }
})

export const {setInitialComment} = commentsSlice.actions

export default commentsSlice.reducer