import { Post } from '../models/Post';
import { EStatuses } from '../enums/Estatuses';
import IPost from '../interfaces/IPost';
import IResponse from '../interfaces/IResponse';
import IPostDto from '../interfaces/IPostDto';
import mongoose, { Mongoose } from 'mongoose'
import IDataBase from '../interfaces/IDataBase';
import IComment from '../interfaces/IComment';
import ICommentDto from '../interfaces/ICommentDto';
import { Comment } from '../models/Comments';
import IUser from '../interfaces/IUser';
import UserModel from '../models/User';

export class Mongo implements IDataBase {
	[x: string]: any;
	private client: Mongoose | null = null

	public close = async (): Promise<void> => {
		if (!this.client) return
		await this.client.disconnect()
	}

	public async init(): Promise<void> {
		this.client = await mongoose.connect(process.env.MONGO_CLIENT_URL || 'mongodb://localhost/myForum')
		console.log('Mongo mongoose is connected')
	}

	public getPosts = async (): Promise<IResponse<IPost[] | undefined>> => {
		try {
			const data = await Post.find()
			const response: IResponse<IPost[]> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Posts found'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}
	public getPostById = async (id: string): Promise<IResponse<IPost | undefined>> => {
		try {

			if (!id || !id.trim() || !mongoose.Types.ObjectId.isValid(id)) {
				const response: IResponse<undefined> = {
					status: EStatuses.NOT_OK,
					result: undefined,
					message: `Invalid post id: ${id}`
				};
				return response;
			}
			const data = await Post.findById(id)

			const response: IResponse<IPost> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Post found'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}
	public addPost = async (postDto: IPostDto): Promise<IResponse<IPost | undefined>> => {
		try {
			const post = new Post(postDto)
			const data = await post.save()
			const response: IResponse<IPost> = {
				status: EStatuses.OK,
				result: data,
				message: 'Post added'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}
	public deletePostById = async (id: string): Promise<IResponse<IPost | undefined>> => {
		try {
			if (!id || !id.trim() || !mongoose.Types.ObjectId.isValid(id)) {
				const response: IResponse<undefined> = {
					status: EStatuses.NOT_OK,
					result: undefined,
					message: `Invalid item id: ${id}`
				};
				return response;
			}
			const data = await Post.findOneAndDelete({ _id: id })
			const response: IResponse<IPost> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Post deleted'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}

	public getComments = async (): Promise<IResponse<IComment[] | undefined>> => {
		try {
			const data = await Comment.find().populate('user').populate('post')
			const response: IResponse<IComment[]> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Comments found'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}
	public getCommentById = async (id: string): Promise<IResponse<IComment | undefined>> => {
		try {
			const data = await Comment.findById(id).populate('user').populate('post');
			const response: IResponse<IComment> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Comment found'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}

	public getCommentByPost = async (post_id: any): Promise<IResponse<IComment[] | undefined>> => {
        try {
            const data = await Comment.find({post: post_id})
            const response: IResponse<IComment[]> = {
                status: EStatuses.OK,
                message: 'Posts comment(s) found',
                result: data as any
            }
            return response
        } catch(err: unknown) {
            const error = err as Error
            const response: IResponse<undefined> = {
                status: EStatuses.NOT_OK,
                message: error.message,
                result: undefined
            }
            return response
        }
    }
<<<<<<< backend/src/repository/mongoose.ts
=======

>>>>>>> backend/src/repository/mongoose.ts
	public addComment = async (commentDto: ICommentDto): Promise<IResponse<IComment | undefined>> => {
		try {
			const comment = new Comment(commentDto)
			const data = await comment.save()
			const response: IResponse<IComment> = {
				status: EStatuses.OK,
				result: data,
				message: 'Comment added'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}
	public deleteCommentById = async (user: IUser, id: string): Promise<IResponse<IComment | undefined>> => {
		try {
			const comment = await Comment.findById(id);
			if (!comment) {
				throw new Error('Comment not found');
			}

			if (comment.user.toString() !== user._id.toString()) {
				throw new Error('You are not authorized to delete this comment!');
			}
			const data = await Comment.findOneAndDelete({ _id: id })
			const response: IResponse<IComment> = {
				status: EStatuses.OK,
				result: data as any,
				message: 'Comment deleted'
			}
			return response
		} catch (err: unknown) {
			const error = err as Error
			const response: IResponse<undefined> = {
				status: EStatuses.NOT_OK,
				result: undefined,
				message: error.message
			}
			return response
		}
	}

};
export const mongo = new Mongo()

