import express, { Router, Request, Response } from 'express'
import multer from 'multer'
import { config } from '../index.config'
import { PostService, postService } from '../services/posts'

const storage = multer.diskStorage({
    destination(req, file, callback) {
        callback(null, 
            config.filePath)
    },
    filename(req, file, callback) {
        callback(null, 
            file.originalname)
    },
})

const upload = multer({storage})

export class PostsController {
    private router: Router
    private service: PostService

    constructor() {
        this.router = express.Router()
        this.router.get('/', this.getPosts)
        this.router.get('/:id', this.getPostById)
        this.router.post('/', upload.single('image'), this.addPost)
        this.router.delete('/:id', this.deletePostById)
        this.service = postService
    }

    public getRouter = (): Router => {
        return this.router
    }
    private getPosts = async (req: Request, res: Response): Promise<void> => {
        const response = await this.service.getPosts()
        res.send(response)
    }

    private getPostById = async (req: Request, res: Response): Promise<void> => {
        const response = await this.service.getPostById(req.params.id)
        res.send(response)
    }

    private addPost = async (req: Request, res: Response): Promise<void> => {
        const post = req.body
        post.image = req.file ? req.file.filename : req.body.image
        const response = await this.service.addPost(post)
        res.send(response)
    }

    private deletePostById = async (req: Request, res: Response): Promise<void> => {
        const response = await this.service.deletePostById(req.params.id)
        res.send(response)
    }
   
}