#!/usr/bin/env bash
# shellcheck disable=SC2086

set -e

echo "User ${whoami}"

echo "================================"
echo "<===== Run apply-update.sh ====>"
echo "================================"

if [[ "$(id -u)" != "0" ]]; then
    echo "This script must be run as root." 1>&2
    exit 1
fi

INSTANCE_NAME=${1}
LATEST_ARTIFACT_NAME=${2}
APPLICATION_ROOT_PATH=${3}
VERSION=$(cat ${APPLICATION_ROOT_PATH}/VERSION)

DB_PASSWORD_ENV=${4}
DB_USERNAME_ENV=${5}
DB_HOST_ENV=${6}
DB_PORT_ENV=${7}

PROJECT_ROOT=${APPLICATION_ROOT_PATH}/application
BACKUP_DIR=~/application_backups

# Create backup directory if not exist
mkdir -p ${BACKUP_DIR}
BACKUP_DATE=$(date +"%Y_%m_%d__%H_%M")
BACKUP_DIR_PATH=${BACKUP_DIR}/${PROJECT_ROOT}_backup_${BACKUP_DATE}_v${VERSION}

DB_DUMP_DIR=${APPLICATION_ROOT_PATH}/dumps/
DB_DUMP_PATH=${DB_DUMP_DIR}backup_${BACKUP_DATE}_v${VERSION}.sql

info() {
    echo -e "\e[33m[Info]     \e[33m$1 \e[39m $(for i in {12..21} ; do echo -en "\e[33;1;${i}m.\e[0m" ; done ; echo)"
}

success() {
    echo -e "\e[32m[Success] \e[32m $1 \e[39m $(for i in {12..21} ; do echo -en "\e[32;1;${i}m.\e[0m" ; done ; echo)"
}

cd ${APPLICATION_ROOT_PATH}

mkdir -p ${DB_DUMP_DIR}

info "# Creating backup"
if [[ -f ${DB_PATH} ]]; then
    info "# Put db to temporary stash"
    cp ${DB_PATH} ${STASH_DB_PATH}
fi

if [ -f ${PROJECT_ROOT} ]; then
    mkdir -p ${BACKUP_DIR_PATH}
    mv ${PROJECT_ROOT} ${BACKUP_DIR_PATH}
fi

info "# Extracting app package to ${PROJECT_ROOT}"
mkdir -p ${PROJECT_ROOT}
tar -xzf ${LATEST_ARTIFACT_NAME} -C ${PROJECT_ROOT}

info "# Setup permission for cache dir and storage"
chmod -R 777 ${PROJECT_ROOT}/storage/
chmod -R 777 ${PROJECT_ROOT}/bootstrap/cache

info "# Set user to .env file"
chown root: ${PROJECT_ROOT}/.env

cd ${PROJECT_ROOT}

info "# Set environments"
cp ${PROJECT_ROOT}/.env ${APPLICATION_ROOT_PATH}/.env
source .env

sed -i -e 's/^#DB_HOST=/DB_HOST=/g' ".env"
sed -i -e 's/^\(DB_HOST=\).*/\1'$([ -n "$DB_HOST_ENV" ] && echo "$DB_HOST_ENV" || echo "$DB_HOST")'/' ${APPLICATION_ROOT_PATH}/.env

sed -i -e 's/^#DB_PORT=/DB_PORT=/g' ".env"
sed -i -e 's/^\(DB_PORT=\).*/\1'$([ -n "$DB_PORT_ENV" ] && echo "$DB_PORT_ENV" || echo "$DB_PORT")'/' ${APPLICATION_ROOT_PATH}/.env

sed -i -e 's/^#DB_USERNAME=/DB_USERNAME=/g' ".env"
sed -i -e 's/^\(DB_USERNAME=\).*/\1'$([ -n "$DB_USERNAME_ENV" ] && echo "$DB_USERNAME_ENV" || echo "$DB_USERNAME")'/' ${APPLICATION_ROOT_PATH}/.env

sed -i -e 's/^#DB_PASSWORD=/DB_PASSWORD=/g' ".env"
sed -i -e 's/^\(DB_PASSWORD=\).*/\1'$([ -n "$DB_PASSWORD_ENV" ] && echo "$DB_PASSWORD_ENV" || echo "$DB_PASSWORD")'/' ${APPLICATION_ROOT_PATH}/.env

cat ${APPLICATION_ROOT_PATH}/.env > ${PROJECT_ROOT}/.env

# info "# Create database backup [${DB_DATABASE}] > ${DB_DUMP_PATH}"
# Здесь требуется настройка доступа к базе данных да пользователя от имени кого будет выполнятся бекап
# mysqldump -u root ${DB_DATABASE} > ${DB_DUMP_PATH}

info "# Set application key"
php artisan key:generate

info "# Optimizing Configuration Loading..."
php artisan config:cache

info "# Optimizing Route Loading"
php artisan route:cache

info "# Optimizing View Loading"
php artisan view:cache

php artisan migrate --seed # Выполним посев

info "# Restart nginx server"
service nginx restart

cd ${APPLICATION_ROOT_PATH}

info "# Destroy archive ${LATEST_ARTIFACT_NAME}"
rm ${LATEST_ARTIFACT_NAME}
info "# Destroy VERSION file"
rm VERSION
info "# Destroy script"
rm apply-update.sh

success "Update successfully!"
